// Chrome-specific popup script
console.log('Chrome popup script loaded');

document.addEventListener('DOMContentLoaded', function() {
  console.log('DOM loaded in popup');
  
  const scrapeBtn = document.getElementById('scrapeBtn');
  const historyBtn = document.getElementById('historyBtn');
  const status = document.getElementById('status');

  if (!scrapeBtn || !status) {
    console.error('Required elements not found in popup');
    return;
  }

  // Check what type of page we're on and show appropriate buttons
  checkPageType();

  async function checkPageType() {
    try {
      const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
      const tab = tabs[0];
      
      if (tab.url.includes('/order-history') && !tab.url.includes('/past-order/')) {
        // On main order history page - show history button
        if (historyBtn) {
          historyBtn.style.display = 'block';
          scrapeBtn.style.display = 'none';
        }
      } else if (tab.url.includes('/past-order/') || tab.url.includes('/order-dashboard/')) {
        // On individual order page - show single order button
        if (historyBtn) {
          historyBtn.style.display = 'none';
          scrapeBtn.style.display = 'block';
        }
      } else {
        // On other page - show both but with different messaging
        updateStatus('Navigate to a PC Express order page to extract data', 'info');
      }
    } catch (error) {
      console.error('Error checking page type:', error);
    }
  }

  // Single order extraction (existing functionality)
  scrapeBtn.addEventListener('click', async function() {
    console.log('Extract button clicked');
    
    scrapeBtn.disabled = true;
    updateStatus('🔄 Extracting order data...', 'info');

    try {
      const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
      const tab = tabs[0];
      
      console.log('Current tab URL:', tab.url);
      
      if (!isSupportedSite(tab.url)) {
        updateStatus('❌ Please navigate to a PC Express order page first', 'error');
        return;
      }
      
      const results = await chrome.scripting.executeScript({
        target: { tabId: tab.id },
        function: scrapePageData
      });

      const scrapedData = results[0].result;
      console.log('Scraped data:', scrapedData);
      
      if (scrapedData && scrapedData.length > 0) {
        const csv = convertToCSV(scrapedData);
        const timestamp = new Date().toISOString().slice(0, 19).replace(/:/g, '-');
        const filename = `pc_express_order_${timestamp}.csv`;
        
        await downloadCSV(csv, filename);
        updateStatus(`✅ Downloaded ${scrapedData.length} items to ${filename}`, 'success');
      } else {
        updateStatus('❌ No order data found on this page', 'error');
      }
    } catch (error) {
      console.error('Scraping error:', error);
      updateStatus(`❌ Error: ${error.message}`, 'error');
    } finally {
      scrapeBtn.disabled = false;
    }
  });

  // Order history extraction (new functionality)
  if (historyBtn) {
    historyBtn.addEventListener('click', async function() {
      console.log('Extract history button clicked');
      
      historyBtn.disabled = true;
      updateStatus('🔄 Extracting order history...', 'info');

      try {
        const tabs = await chrome.tabs.query({ active: true, currentWindow: true });
        const tab = tabs[0];
        
        console.log('Current tab URL:', tab.url);
        
        if (!isSupportedSite(tab.url)) {
          updateStatus('❌ Please navigate to a PC Express order history page first', 'error');
          return;
        }
        
        const results = await chrome.scripting.executeScript({
          target: { tabId: tab.id },
          function: scrapeOrderHistory
        });

        const historyData = results[0].result;
        console.log('History data:', historyData);
        
        if (historyData && historyData.length > 0) {
          const csv = convertToCSV(historyData);
          const timestamp = new Date().toISOString().slice(0, 19).replace(/:/g, '-');
          const filename = `pc_express_order_history_${timestamp}.csv`;
          
          await downloadCSV(csv, filename);
          updateStatus(`✅ Downloaded ${historyData.length} orders to ${filename}`, 'success');
        } else {
          updateStatus('❌ No order history found on this page', 'error');
        }
      } catch (error) {
        console.error('History scraping error:', error);
        updateStatus(`❌ Error: ${error.message}`, 'error');
      } finally {
        historyBtn.disabled = false;
      }
    });
  }

  function updateStatus(message, type) {
    status.textContent = message;
    status.className = type;
  }

  function isSupportedSite(url) {
    const supportedDomains = [
      'loblaws.ca',
      'provigo.ca', 
      'nofrills.ca',
      'superstore.ca',
      'zehrs.ca',
      'fortinos.ca',
      'valu-mart.ca',
      'yourindependentgrocer.ca',
      'maxi.ca'
    ];
    
    return supportedDomains.some(domain => url.includes(domain));
  }

  function convertToCSV(data) {
    if (!data || data.length === 0) return '';
    
    const headers = Object.keys(data[0]);
    const csvRows = [headers.join(',')];
    
    for (const row of data) {
      const values = headers.map(header => {
        const value = (row[header] || '').toString();
        // Escape quotes and wrap in quotes if contains comma, quote, or newline
        return /[",\n\r]/.test(value) ? `"${value.replace(/"/g, '""')}"` : value;
      });
      csvRows.push(values.join(','));
    }
    
    return csvRows.join('\n');
  }

  async function downloadCSV(csvContent, filename) {
    try {
      console.log('Starting download:', filename);
      console.log('CSV content preview:', csvContent.substring(0, 100) + '...');
      
      // Ensure filename has .csv extension
      if (!filename.toLowerCase().endsWith('.csv')) {
        filename += '.csv';
      }
      
      // Create blob with BOM for proper UTF-8 encoding
      const BOM = '\uFEFF';
      const blob = new Blob([BOM + csvContent], { 
        type: 'text/csv;charset=utf-8' 
      });
      
      console.log('Blob created, size:', blob.size, 'bytes');
      
      // Use Chrome downloads API
      const url = URL.createObjectURL(blob);
      console.log('Blob URL created:', url);
      
      const downloadId = await chrome.downloads.download({
        url: url,
        filename: filename,
        saveAs: false  // Don't show save dialog, use default location
      });
      
      console.log('Download started with ID:', downloadId);
      
      // Clean up after delay
      setTimeout(() => {
        URL.revokeObjectURL(url);
        console.log('Blob URL cleaned up');
      }, 3000);
      
    } catch (error) {
      console.error('Download failed:', error);
      throw new Error('Download failed. Please check Chrome download settings and permissions.');
    }
  }
});

// Scraping function that runs in the webpage context - Individual orders
function scrapePageData() {
  console.log('Starting to scrape page data...');
  const data = [];
  
  // Get order description from header
  const descriptionElement = document.querySelector('h1.order-dashboard-hero-time-slot');
  const description = descriptionElement?.textContent?.trim() || '';
  console.log('Found description:', description);
  
  // Extract date from description (e.g., "In-store purchase, May 23, 7:31pm")
  let date = '';
  if (description) {
    const dateMatch = description.match(/([A-Za-z]+\s+\d+)(?:,\s*(\d+:\d+[ap]m))?/);
    if (dateMatch) {
      const currentYear = new Date().getFullYear();
      const dateStr = `${dateMatch[1]}, ${currentYear}`;
      const timeStr = dateMatch[2] || '12:00am';
      
      try {
        const parsedDate = new Date(`${dateStr} ${timeStr}`);
        date = parsedDate.toISOString();
      } catch (e) {
        console.warn('Date parsing failed, using current time');
        date = new Date().toISOString();
      }
    } else {
      date = new Date().toISOString();
    }
  }
  
  // Get all product entries
  const cartEntries = document.querySelectorAll('.cart-entry__desktop-content');
  console.log('Found cart entries:', cartEntries.length);
  
  cartEntries.forEach((entry, index) => {
    console.log(`Processing entry ${index + 1}`);
    
    // Get product name
    const productNameElement = entry.querySelector('.cart-entry__content--product-name');
    const productName = productNameElement?.textContent?.trim() || '';
    
    // Get quantity from input value
    const quantityInput = entry.querySelector('.cart-entry__content--quantity-selector input.cart-entry__content__quantity');
    const quantity = quantityInput?.value?.trim() || '';
    
    // Get price from the price element
    const priceElement = entry.querySelector('.price__value.cart-entry__content__price__value');
    let price = priceElement?.textContent?.trim() || '';
    
    // Clean up price (remove $ symbol)
    price = price.replace(/^\$/, '');
    
    console.log(`Entry ${index + 1}:`, { productName, quantity, price });
    
    // Only add rows with valid product data (skip items with price 0 or empty)
    if (productName && quantity && price && price !== '0' && price !== '') {
      data.push({
        description: description,
        date: date,
        product_name: productName,
        quantity: quantity,
        price: price
      });
    }
  });
  
  console.log('Scraped data:', data);
  return data;
}

// Order history scraping function that runs in the webpage context
function scrapeOrderHistory() {
  console.log('Starting to scrape order history...');
  const data = [];
  
  // Get all order history items
  const orderItems = document.querySelectorAll('.account-order-history-past-orders-delivery-list-item');
  console.log('Found order items:', orderItems.length);
  
  orderItems.forEach((item, index) => {
    console.log(`Processing order ${index + 1}`);
    
    // Extract store name from the location element
    const locationElement = item.querySelector('.account-order-history-past-orders-delivery-list-item__details__location');
    const store = locationElement?.textContent?.trim() || '';
    
    // Extract date
    const dateElement = item.querySelector('.account-order-history-past-orders-delivery-list-item__details__date');
    const dateText = dateElement?.textContent?.trim() || '';
    
    // Convert date to ISO format
    let date = '';
    if (dateText) {
      try {
        // Parse dates like "May 23, 2025" or "Dec 31, 2023"
        const parsedDate = new Date(dateText);
        date = parsedDate.toISOString().split('T')[0]; // Just the date part (YYYY-MM-DD)
      } catch (e) {
        console.warn('Date parsing failed for:', dateText);
        date = dateText; // Keep original if parsing fails
      }
    }
    
    // Extract purchase total
    const priceElement = item.querySelector('.account-order-history-past-orders-delivery-list-item__price');
    let purchaseTotal = priceElement?.textContent?.trim() || '';
    
    // Clean up price (remove $ symbol)
    purchaseTotal = purchaseTotal.replace(/^\$/, '');
    
    console.log(`Order ${index + 1}:`, { store, date, purchaseTotal });
    
    // Only add rows with valid data
    if (store && date && purchaseTotal) {
      data.push({
        store: store,
        date: date,
        purchase_total: purchaseTotal
      });
    }
  });
  
  console.log('Order history data:', data);
  return data;
}